################################################################################

#### Author: Maria Barrera-Cruz.

#### License: Permission is granted to use and adapt this code. Please 
#### acknowledge authorship when appropriate.

#### This code allows full reproducibility of the article "Evolution of Harvesting Technology:
#### A study of diversity from lithic and botanical remains in Neolithic Iberia (8th-7th millennium cal. BP)", 
#### by M. Barrera-Cruz, O. García-Puchol, A. Cortell-Nicolau and D. García-Rivero 


########################################
## 200 years Time window creation ######
########################################

##Load the library
##################
library(xlsx)
library(rcarbon)
library(vegan)
library(cluster)
library(dplyr)

#Load the data. 
###############
#[Here, dates ready for time-bin creation (filtered and combined).]
dates <- read.csv(file = "datesfiltered2.csv", header = TRUE, sep = ",", encoding = "UTF-8") 
dates <- dates[,-1]

#Calibrate
###########
#Extract bp and sds
bp <- dates[,12]
sds <- dates[,13]
#Create a column to define the cal.curve used according to the sample
dates$calCurves <- ifelse(
  dates$Muestra == "Marine",
  "marine20", "intcal20")   
#Calibrate
datacal <- rcarbon::calibrate(bp, sds, 
                              calCurves = c(dates[,19])) 

#Time-bins by PrDens
####################

upbond <- seq(4000, 7600, by = 200) 
lowbond <- seq(3801, 7601, by = 200)

#-------------------------------
v7600.7401 <- as.data.frame(matrix(NA,nrow(dates), ncol(dates)))
names(v7600.7401) <- c(colnames(dates))

for (i in 1:400){
  It <- as.character(i)
  a <- sum(datacal$grids[[It]]$PrDens[datacal$grids[[It]]$calBP %in% c(upbond[19]:lowbond[19])])
  if(a >= 0.25) {
    v7600.7401[i,] <- dates[i,]
  } else {print(i)}
}
v7600.7401 <- v7600.7401[!is.na(v7600.7401$ID_Fase),]

#-------------------------------
v7400.7201 <- as.data.frame(matrix(NA,nrow(dates), ncol(dates)))
names(v7400.7201) <- c(colnames(dates))

for (i in 1:400){
  It <- as.character(i)
  a <- sum(datacal$grids[[It]]$PrDens[datacal$grids[[It]]$calBP %in% c(upbond[18]:lowbond[18])])
  if(a >= 0.25) {
    v7400.7201[i,] <- dates[i,]
  } else {print(i)}
}
v7400.7201 <- v7400.7201[!is.na(v7400.7201$ID_Fase),]

#-------------------------------
v7200.7001 <- as.data.frame(matrix(NA,nrow(dates), ncol(dates)))
names(v7200.7001) <- c(colnames(dates))

for (i in 1:400){
  It <- as.character(i)
  a <- sum(datacal$grids[[It]]$PrDens[datacal$grids[[It]]$calBP %in% c(upbond[17]:lowbond[17])])
  if(a >= 0.25) {
    v7200.7001[i,] <- dates[i,]
  } else {print(i)}
}
v7200.7001 <- v7200.7001[!is.na(v7200.7001$ID_Fase),]

#-------------------------------
v7000.6801 <- as.data.frame(matrix(NA,nrow(dates), ncol(dates)))
names(v7000.6801) <- c(colnames(dates))

for (i in 1:400){
  It <- as.character(i)
  a <- sum(datacal$grids[[It]]$PrDens[datacal$grids[[It]]$calBP %in% c(upbond[16]:lowbond[16])])
  if(a >= 0.25) {
    v7000.6801[i,] <- dates[i,]
  } else {print(i)}
}
v7000.6801 <- v7000.6801[!is.na(v7000.6801$ID_Fase),]

#-------------------------------
v6800.6601 <- as.data.frame(matrix(NA,nrow(dates), ncol(dates)))
names(v6800.6601) <- c(colnames(dates))

for (i in 1:400){
  It <- as.character(i)
  a <- sum(datacal$grids[[It]]$PrDens[datacal$grids[[It]]$calBP %in% c(upbond[15]:lowbond[15])])
  if(a >= 0.25) {
    v6800.6601[i,] <- dates[i,]
  } else {print(i)}
}

v6800.6601 <- v6800.6601[!is.na(v6800.6601$ID_Fase),]

#-------------------------------
v6600.6401 <- as.data.frame(matrix(NA,nrow(dates), ncol(dates)))
names(v6600.6401) <- c(colnames(dates))

for (i in 1:400){
  It <- as.character(i)
  a <- sum(datacal$grids[[It]]$PrDens[datacal$grids[[It]]$calBP %in% c(upbond[14]:lowbond[14])])
  if(a >= 0.25) {
    v6600.6401[i,] <- dates[i,]
  } else {print(i)}
}

v6600.6401 <- v6600.6401[!is.na(v6600.6401$ID_Fase),]


###############################################
#### Testing differences in blade measures ####
###############################################

##Load the library
##################
library(tidyverse)
library(car)
library(rstatix)
library(ggplot2)
library(ggstatsplot)
library(ggpubr)
library(pcal)

##Load the data
####################################
Blades <- read.csv("BladeJLS.csv", header = TRUE, sep = ",", encoding = "UTF-8") 


## Area
####################################

##Testing normal distribution by area
shapiro.test(dplyr::filter(.data=Blades, Area == "South", Lustre == "Glossy blades")$Width)
shapiro.test(dplyr::filter(.data=Blades, Area == "Xuquer", Lustre == "Glossy blades")$Width)
shapiro.test(dplyr::filter(.data=Blades, Area == "Ebro", Lustre == "Glossy blades")$Width)
shapiro.test(dplyr::filter(.data=Blades, Area == "Northeast", Lustre == "Glossy blades")$Width)
shapiro.test(dplyr::filter(.data=Blades, Area == "Interior", Lustre == "Glossy blades")$Width)

#Testing homocedasticity
bartlett.test(Width ~ Area, dplyr::filter(.data = Blades, Lustre == "Glossy blades")) 

#Testing comparisons and saving results into a graphic .jpg
p1 <-  ggbetweenstats(
  data = dplyr::filter(.data=Blades, Lustre == "Glossy blades"),
  type = "p",
  var.equal = FALSE,
  x = Area,
  y = Width,
  results.subtitle = FALSE,
  title = "Glossy blades' width between Areas",
  pairwise.comparisons = TRUE)

ggsave("BARRERACRUZ_etal_Figure4.jpg", plot = p1, width = 23, height = 14, units = "cm", dpi = 600)


# Parallel by Area
#################################

##Testing normal distribution in parallel assemblages
table(Blades$Area, Blades$Insertion)
shapiro.test(dplyr::filter(.data=Blades, Area == "Xuquer", Insertion == "Parallel")$Width)
shapiro.test(dplyr::filter(.data=Blades, Area == "Interior", Insertion == "Parallel")$Width)
shapiro.test(dplyr::filter(.data=Blades, Area == "Northeast", Insertion == "Parallel")$Width)
shapiro.test(dplyr::filter(.data=Blades, Area == "Ebro", Insertion == "Parallel")$Width)

##Testing homocedasticity
bartlett.test(Width ~ Area, dplyr::filter(.data = Blades, Insertion == "Parallel")) 

#Testing comparisons and saving results into a graphic .jpg
p2 <-  ggbetweenstats(
  data = dplyr::filter(.data = Blades, Insertion == "Parallel"),
  type = "p",
  var.equal = FALSE,
  x = Area,
  y = Width,
  results.subtitle = FALSE,
  title = "Parallel's width between Areas",
  pairwise.comparisons = TRUE)

ggsave("BARRERACRUZ_etal-Figure5.jpg", plot = p2, width = 23, height = 14, units = "cm", dpi = 600)


### Oblique multiple Xuquer vs Oblique multiple South
#######################################################

##Testing normal distribution 
shapiro.test(dplyr::filter(.data=Blades, Area == "South", Insertion == "Oblique multiple")$Width)
shapiro.test(dplyr::filter(.data=Blades, Area == "Xuquer", Insertion == "Oblique multiple")$Width)

##Testing homocedasticity
bartlett.test(Width ~ Area, dplyr::filter(.data = Blades, 
                                          Area == "South" & Insertion == "Oblique multiple" |
                                            Area == "Xuquer" & Insertion == "Oblique multiple")) 

#Testing comparisons and saving results into a graphic .jpg
p6 <-  ggbetweenstats(
  data = dplyr::filter(.data = Blades, 
                       Area == "South" & Insertion == "Oblique multiple" |
                         Area == "Xuquer" & Insertion == "Oblique multiple"),
  type = "p",
  var.equal = TRUE,
  x = Area,
  y = Width,
  #results.subtitle = FALSE,
  title = "Oblique multiple in South and Xuquer",
  #subtitle = "non-parametric test",
  #pairwise.comparisons = TRUE,
  bf.message = FALSE,
)


ggsave("BARRERACRUZ_etal-Figure6.jpg", plot = p6, width = 22, height = 12, units = "cm", dpi = 600)


### Parallel Xuquer vs Oblique multiple South
####################################################

##Testing normal distribution 
shapiro.test(dplyr::filter(.data=Blades, Area == "South", Insertion == "Oblique multiple")$Width)
shapiro.test(dplyr::filter(.data=Blades, Area == "Xuquer", Insertion == "Parallel")$Width)

##Testing homocedasticity
bartlett.test(Width ~ Insertion, dplyr::filter(.data = Blades, 
                                               Area == "South" & Insertion == "Oblique multiple" |
                                                 Area == "Xuquer" & Insertion == "Parallel")) 

#Testing comparisons and saving results into a graphic .jpg
p5a <-  ggbetweenstats(
  data = dplyr::filter(.data = Blades, 
                       Area == "South" & Insertion == "Oblique multiple" |
                         Area == "Xuquer" & Insertion == "Parallel"),
  type = "p",
  var.equal = TRUE,
  x = Insertion,
  y = Width,
  #results.subtitle = FALSE,
  title = "Oblique multiple in South and Parallel in Xuquer",
  #subtitle = "non-parametric test",
  #pairwise.comparisons = TRUE,
  bf.message = FALSE
)

## Without Benamer III
##############################

##Testing normal distribution 
shapiro.test(dplyr::filter(.data=Blades, Area == "South", Insertion == "Oblique multiple")$Width)
shapiro.test(dplyr::filter(.data=Blades, Area == "Xuquer", Insertion == "Parallel", Chronology != "6600-6400")$Width)

##Testing homocedasticity
bartlett.test(Width ~ Insertion, dplyr::filter(.data = Blades, 
                                               Area == "South" & Insertion == "Oblique multiple" |
                                                 Area == "Xuquer" & Insertion == "Parallel" & Chronology != "6600-6400")) 

#Testing comparisons and saving results into a graphic .jpg
p5b <-  ggbetweenstats(
  data = dplyr::filter(.data = Blades, 
                       Area == "South" & Insertion == "Oblique multiple" |
                         Area == "Xuquer" & Insertion == "Parallel" & Chronology != "6600-6400"),
  type = "p",
  var.equal = TRUE,
  x = Insertion,
  y = Width,
  #results.subtitle = FALSE,
  title = "Oblique multiple in South and Parallel in Xuquer -excluding Benamer III-",
  #subtitle = "non-parametric test",
  #pairwise.comparisons = TRUE,
  bf.message = FALSE
)


Fig.p5 <- ggarrange(p5a, p5b,  
                    ncol = 1, nrow = 2
                    #, legend.grob = leg.Fig4, legend = "right")
)

ggsave("BARRERACRUZ_etal-Figure7.jpg", plot = Fig.p5, width = 23, height = 14, units = "cm", dpi = 600)

# Glossy blades into blade production. Xuquer and South Areas
#################################################################

### South
####################################################
##Testing normal distribution 
shapiro.test(dplyr::filter(.data=Blades, Area == "South", Insertion == "Oblique multiple")$Width) 
shapiro.test(dplyr::filter(.data=Blades, Area == "South", Insertion == "Other blades")$Width) 

##Testing homocedasticity
bartlett.test(Width ~ Insertion, dplyr::filter(.data = Blades, Area == "South")) 

#Testing comparisons and saving results into a graphic .jpg
p4 <-  ggbetweenstats(
  data = dplyr::filter(.data = Blades, Area == "South"),
  type = "np",
  var.equal = TRUE,
  x = Insertion,
  y = Width,
  #results.subtitle = FALSE,
  title = "Blades' width in South",
  bf.message = FALSE,
  #subtitle = "non-parametric test",
  pairwise.comparisons = TRUE)


ggsave("BARRERACRUZ_etal-Figure8.jpg", plot = p4, width = 22, height = 12, units = "cm", dpi = 600)

### XUQUER
###########
##Testing normal distribution in parallel assemblages
shapiro.test(dplyr::filter(.data=Blades, Area == "Xuquer", Insertion == "Parallel")$Width)
shapiro.test(dplyr::filter(.data=Blades, Area == "Xuquer", Insertion == "Oblique multiple")$Width)
shapiro.test(dplyr::filter(.data=Blades, Area == "Xuquer", Insertion == "Other blades")$Width) 

#Testing comparisons and saving results into a graphic .jpg
p3a <- ggbetweenstats(
  data = dplyr::filter(.data = Blades, Area == "Xuquer"),
  type = "np",
  x = Insertion,
  y = Width,
  #results.subtitle = FALSE,
  title = "Blades' width in Xuquer",
  #subtitle = "non-parametric test",
  pairwise.comparisons = TRUE)

table(Blades$Area, Blades$Chronology)

p3b <- ggbetweenstats(
  data = dplyr::filter(.data = Blades, Area == "Xuquer", Chronology != "6600-6400"),
  type = "np",
  x = Insertion,
  y = Width,
  #results.subtitle = FALSE,
  title = "Blades' width in Xuquer (excluding Benàmer III)",
  #subtitle = "non-parametric test",
  pairwise.comparisons = TRUE)

Fig.p9 <- ggarrange(p3a, p3b,  
                    ncol = 1, nrow = 2, widths = c(1, -0.1, 1)
                    #, legend.grob = leg.Fig4, legend = "right")
)

ggsave("BARRERACRUZ_etal-Figure9.jpg", plot = Fig.p9, width = 23, height = 14, units = "cm", dpi = 600)


### Blade production Xuquer and South
#######################################################
##Testing normal distribution 
shapiro.test(dplyr::filter(.data=Blades, Area == "South")$Width)
shapiro.test(dplyr::filter(.data=Blades, Area == "Xuquer")$Width)

##Testing homocedasticity
leveneTest(Width ~ Area, dplyr::filter(.data = Blades, 
                                       Area == "South" |
                                         Area == "Xuquer")) 

#Testing comparisons and saving results into a graphic .jpg
p7 <-  ggbetweenstats(
  data = dplyr::filter(.data = Blades, 
                       Area == "South" |
                         Area == "Xuquer"),
  type = "np",
  var.equal = FALSE,
  x = Area,
  y = Width,
  #results.subtitle = FALSE,
  title = "Blades in South and Xuquer",
  #subtitle = "non-parametric test",
  #pairwise.comparisons = TRUE
)

#Excluding Benàmer III

##Testing normal distribution 
shapiro.test(dplyr::filter(.data=Blades, Area == "South")$Width)
shapiro.test(dplyr::filter(.data=Blades, Area == "Xuquer", Chronology != "6600-6400")$Width)

##Testing homocedasticity
leveneTest(Width ~ Area, dplyr::filter(.data = Blades, 
                                       Area == "South" |
                                         Area == "Xuquer" & Chronology != "6600-6400")) 

#Testing comparisons and saving results into a graphic .jpg
Bladesp7b <- dplyr::filter(.data = Blades, 
                           Area == "South" |
                             Area == "Xuquer", Chronology != "6600-6400")

p7b <-  ggbetweenstats(
  data = Bladesp7b,
  type = "np",
  var.equal = FALSE,
  x = Area,
  y = Width,
  #results.subtitle = FALSE,
  title = "Blades in South and Xuquer -excluding Benàmer III-",
  #subtitle = "non-parametric test",
  #pairwise.comparisons = TRUE
)

Fig.p10 <- ggarrange(p7, p7b,  
                    ncol = 1, nrow = 2, widths = c(1, -0.1, 1)
                    #, legend.grob = leg.Fig4, legend = "right")
)
ggsave("BARRERACRUZ_etal-Figure10.jpg", plot = Fig.p10, width = 22, height = 12, units = "cm", dpi = 600)


### Variance between inserted types
#######################################################
##Testing normal distribution 
shapiro.test(dplyr::filter(.data=Blades, Insertion == "Oblique multiple")$Width)
shapiro.test(dplyr::filter(.data=Blades, Insertion == "Oblique simple")$Width)
shapiro.test(dplyr::filter(.data=Blades, Insertion == "Parallel")$Width)

##Testing homocedasticity

leveneTest(Width ~ Insertion, dplyr::filter(.data = Blades,
                                            Insertion == "Oblique simple" |
                                              Insertion == "Oblique multiple")) #Oblique simple vs Oblique multiple

leveneTest(Width ~ Insertion, dplyr::filter(.data = Blades,
                                            Insertion == "Oblique simple" |
                                              Insertion == "Parallel")) #Oblique simple vs Parallel

leveneTest(Width ~ Insertion, dplyr::filter(.data = Blades,
                                            Insertion == "Oblique multiple" |
                                              Insertion == "Parallel")) #Oblique multiple vs Parallel


### Glossy blade measures by insertion type and Area
#######################################################
library(ggplot2)
library(ggridges)

p12a <- ggplot(data = dplyr::filter(.data = Blades, 
                            Insertion != "Other blades"), 
       aes(x = Width, y = Area, fill = Insertion)) +
#  geom_boxplot(width = .24,
#               outlier.shape = NA,
#               alpha = 0.9)+
  geom_density_ridges(jittered_points = TRUE,
                      point_shape = "|", point_size = 2, size = 0.25,
                      position = position_points_jitter(height = 0), alpha = 0.6, scale = 0.8) + 
  scale_fill_manual(values=c("#5a7172", "#fcffb6", "#1de5ff")) +
  theme_bw() + #no color grid
  labs(title = "7600 - 6400 cal. BP")+
  theme(text = element_text(size = 12))+
  theme(legend.key.size = unit(1, units = "cm"))+
  theme_classic()


p12b <- ggplot(data = dplyr::filter(.data = Blades, 
                            Insertion != "Other blades"), 
       aes(x = Width, y = Insertion, fill = Insertion)) +
  geom_boxplot(width = .24,
               outlier.shape = NA,
               alpha = 0.9)+
  geom_density_ridges(jittered_points = TRUE,
                      point_shape = "|", point_size = 2, size = 0.25,
                      position = position_points_jitter(height = 0), alpha = 0.6, scale = 0.8) + 
  scale_fill_manual(values=c("#5a7172", "#fcffb6", "#1de5ff")) +
  theme_bw() + #no color grid
  #labs(title = "Insertion")+
  theme(text = element_text(size = 12))+
  theme(legend.key.size = unit(1, units = "cm"))+
  theme_classic()

p12 <- ggpubr::ggarrange(p12a, p12b, ncol = 1, nrow = 2, common.legend = TRUE, align = "v", labels = c('A', 'B'))

ggsave("BARRERACRUZ_etal-Figure12.jpg", plot = p12, width = 16, height = 22, units = "cm", dpi = 600)


#########################################
## Analysis of vegetal remains ##########
#########################################

##Load the library
##################
library(tcltk)
library(vegan)
library(permute)
library(lattice)
library(BiodiversityR)
library(ggplot2)

#Load the data
###############
crop7674 <- read.csv("crop7674.csv", header = TRUE, encoding = "UTF-8", row.names = 1)
crop7472 <- read.csv("crop7472.csv", header = TRUE, encoding = "UTF-8", row.names = 1)
crop7268 <- read.csv("crop7268.csv", header = TRUE, encoding = "UTF-8", row.names = 1)
crop6864 <- read.csv("crop6864.csv", header = TRUE, encoding = "UTF-8", row.names = 1)


#############################
## Renyi diversity profiles
#############################

#Transform the data
###############################################################################
crop7674.renyi <- crop7674[,-c(10,11)]
crop7674.renyi[] <- lapply(crop7674.renyi, function(x) as.numeric(x))

crop7674.env <- crop7674[,c(10,11)]
table(crop7674.env$Area)

crop7674.env$Area <- factor(crop7674$Area,
                                 levels = c("South", "Xuquer", "Northeast"),
                                 labels = c("South", "Xuquer", "Northeast"))
str(crop7674.env)
#To compare diversity between subsets of the dataset:
Renyi.7674 <- renyicomp(crop7674.renyi, y=crop7674.env, factor='Area', 
                        scales=c(0, 0.25, 0.5, 1, 2, 4, 8, Inf), permutations=1000, plotit=F)

renyi.long7674 <- renyicomp.long(Renyi.7674, label.freq=1)
head(renyi.long7674)

####
crop7472.renyi <- crop7472[,-c(10,11)]
crop7472.renyi[] <- lapply(crop7472.renyi, function(x) as.numeric(x))

crop7472.env <- crop7472[,c(10,11)]
table(crop7472.env$Area)

crop7472.env$Area <- factor(crop7472$Area,
                                 levels = c("South", "Xuquer", "Ebro", "Interior", "Northeast"),
                                 labels = c("South", "Xuquer", "Ebro", "Interior", "Northeast"))
str(crop7472.env)
#To compare diversity between subsets of the dataset:
Renyi.7472 <- renyicomp(crop7472.renyi, y=crop7472.env, factor='Area', 
                        scales=c(0, 0.25, 0.5, 1, 2, 4, 8, Inf), permutations=1000, plotit=F)

renyi.long7472 <- renyicomp.long(Renyi.7472, label.freq=1)
head(renyi.long7472)

####
crop7268.renyi <- crop7268[,-c(10,11)]
crop7268.renyi[] <- lapply(crop7268.renyi, function(x) as.numeric(x))

crop7268.env <- crop7268[,c(10,11)]
table(crop7268.env$Area)

crop7268.env$Area <- factor(crop7268$Area,
                                 levels = c("South", "Xuquer", "Ebro", "Interior", "Northeast"),
                                 labels = c("South", "Xuquer", "Ebro", "Interior", "Northeast"))
str(crop7268.env)
#To compare diversity between subsets of the dataset:
Renyi.7268 <- renyicomp(crop7268.renyi, y=crop7268.env, factor='Area', 
                        scales=c(0, 0.25, 0.5, 1, 2, 4, 8, Inf), permutations=1000, plotit=F)

renyi.long7268 <- renyicomp.long(Renyi.7268, label.freq=1)
head(renyi.long7268)

####
crop6864.renyi <- crop6864[,-c(10,11)]
crop6864.renyi[] <- lapply(crop6864.renyi, function(x) as.numeric(x))

crop6864.env <- crop6864[,c(10,11)]
table(crop6864.env$Area)

crop6864.env$Area <- factor(crop6864$Area,
                                 levels = c("South", "Xuquer",  "Interior", "Northeast"),
                                 labels = c("South", "Xuquer",  "Interior", "Northeast"))
str(crop6864.env)
#To compare diversity between subsets of the dataset:
Renyi.6864 <- renyicomp(crop6864.renyi, y=crop6864.env, factor='Area', 
                        scales=c(0, 0.25, 0.5, 1, 2, 4, 8, Inf), permutations=1000, plotit=F)

renyi.long6864 <- renyicomp.long(Renyi.6864, label.freq=1)
head(renyi.long6864)

#Plot Renyi diversity profiles
###############################################################################
BioR.theme <- theme(
  panel.background = element_blank(),
  panel.border = element_blank(),
  panel.grid = element_blank(),
  axis.line = element_line("gray25"),
  text = element_text(size = 12),
  axis.text = element_text(size = 10, colour = "gray25"),
  axis.title = element_text(size = 14, colour = "gray25"),
  legend.title = element_text(size = 14),
  legend.text = element_text(size = 14),
  legend.key = element_blank())

renyiplot.7674 <- 
  ggplot(data=renyi.long7674, aes(x=Scales, y=Diversity, ymax=UPR, ymin=LWR)) + 
  scale_x_discrete() +
  scale_y_continuous(sec.axis = dup_axis(labels=NULL, name=NULL)) +
  geom_line(data=renyi.long7674, 
            aes(x=Obs, colour=Grouping), 
            size=1) +
  geom_point(data=subset(renyi.long7674, labelit==TRUE), 
             aes(colour=Grouping, shape=Grouping), 
             size=3) +
  BioR.theme +
  scale_colour_manual(values = c("#00A087FF", "#3C5488FF", "#B09C85FF")) +
  scale_shape_manual(values=c(15,14,13))+
  labs(x=expression(alpha), y = "Diversity", colour = "Area", shape = "Area") +
  ggtitle("7600 - 7400 cal. BP")

####
renyiplot.7472 <- 
  ggplot(data=renyi.long7472, aes(x=Scales, y=Diversity, ymax=UPR, ymin=LWR)) + 
  scale_x_discrete() +
  scale_y_continuous(sec.axis = dup_axis(labels=NULL, name=NULL)) +
  geom_line(data=renyi.long7472, 
            aes(x=Obs, colour=Grouping), 
            size=1) +
  geom_point(data=subset(renyi.long7472, labelit==TRUE), 
             aes(colour=Grouping, shape=Grouping), 
             size=3) +
 BioR.theme +
  scale_colour_manual(values = c("#E64B35FF", "#4DBBD5FF", "#00A087FF", "#3C5488FF", "#B09C85FF")) +
  scale_shape_manual(values=c(17,16,15,14,13))+
  labs(x=expression(alpha), y = "Diversity", colour = "Area", shape = "Area")+
  ggtitle("7400 - 7200 cal. BP")

####
renyiplot.7268 <- 
  ggplot(data=renyi.long7268, aes(x=Scales, y=Diversity, ymax=UPR, ymin=LWR)) + 
  scale_x_discrete() +
  scale_y_continuous(sec.axis = dup_axis(labels=NULL, name=NULL)) +
  geom_line(data=renyi.long7268, 
            aes(x=Obs, colour=Grouping), 
            size=1) +
  geom_point(data=subset(renyi.long7268, labelit==TRUE), 
             aes(colour=Grouping, shape=Grouping), 
             size=3) +
  BioR.theme +
  scale_colour_manual(values = c("#E64B35FF", "#4DBBD5FF", "#00A087FF", "#3C5488FF", "#B09C85FF")) +
  scale_shape_manual(values=c(17,16,15,14,13))+
  labs(x=expression(alpha), y = "Diversity", colour = "Area", shape = "Area") +
  ggtitle("7200 - 6800 cal. BP")

####
renyiplot.6864 <- 
  ggplot(data=renyi.long6864, aes(x=Scales, y=Diversity, ymax=UPR, ymin=LWR)) + 
  scale_x_discrete() +
  scale_y_continuous(sec.axis = dup_axis(labels=NULL, name=NULL)) +
  geom_line(data=renyi.long6864, 
            aes(x=Obs, colour=Grouping), 
            size=1) +
  geom_point(data=subset(renyi.long6864, labelit==TRUE), 
             aes(colour=Grouping, shape=Grouping), 
             size=3) +
  BioR.theme +
  scale_colour_manual(values = c("#4DBBD5FF", "#00A087FF", "#3C5488FF", "#B09C85FF")) +
  scale_shape_manual(values=c(16,15,14,13))+
  labs(x=expression(alpha), y = "Diversity", colour = "Area", shape = "Area") +
  ggtitle("6800 - 6400 cal. BP")

renyi.diacronico <- ggpubr::ggarrange(renyiplot.7674, renyiplot.7472, renyiplot.7268, renyiplot.6864,
                                      ncol = 1, nrow = 4, common.legend = TRUE)


#############################
## 100% Barplots
#############################
library(reshape)
library(data.table)

#Transform the data
###############################################################################
veg.v7600.7401.long <- melt(setDT(crop7674), id.vars = c("Contexto","Area"), variable.name = "specie") 
####
veg.v7400.7201.long <- melt(setDT(crop7472), id.vars = c("Contexto","Area"), variable.name = "specie") 
####
veg.v7200.6801.long <- melt(setDT(crop7268), id.vars = c("Contexto","Area"), variable.name = "specie") 
####
veg.v6800.6401.long <- melt(setDT(crop6864), id.vars = c("Contexto","Area"), variable.name = "specie") 

#Plot 100% barplots
###############################################################################
plot7674 <- ggplot(veg.v7600.7401.long, aes(fill=specie, y=value, x=Area)) + 
  geom_bar(position="fill", stat="identity") + #We use position="fill" to calculate % instead of frequency
  scale_fill_manual(values = c("azure3", "darkseagreen","goldenrod","goldenrod1","lemonchiffon",
                               "darkolivegreen3","goldenrod4", "darkseagreen2","darkseagreen3")) + 
  ggtitle("7600-7400 cal. BP") +
  theme(axis.text.x = element_text(angle = 90)) +
  theme_bw()

####
plot7472 <- ggplot(veg.v7400.7201.long, aes(fill=specie, y=value, x=Area)) + 
  geom_bar(position="fill", stat="identity") +
  scale_fill_manual(values = c("azure3", "darkseagreen","goldenrod","goldenrod1","lemonchiffon",
                               "darkolivegreen3","goldenrod4","darkseagreen2","darkseagreen3")) + 
  #scale_fill_brewer(palette = "Set3") +
  ggtitle("7400-7200 cal. BP") +
  theme(axis.title.x = element_blank(),
        axis.text.x  = element_text(angle = 90)) +
  theme_bw()

####
plot7268 <- ggplot(veg.v7200.6801.long, aes(fill=specie, y=value, x=Area)) + 
  geom_bar(position="fill", stat="identity") +
  scale_fill_manual(values = c("azure3", "darkseagreen","goldenrod","goldenrod1","lemonchiffon",
                               "darkolivegreen3","goldenrod4","darkseagreen2","darkseagreen3")) + 
  ggtitle("7200-6800 cal. BP") +
  theme(axis.title.x = element_blank(),
        axis.text.x  = element_text(angle = 90)) +
  theme_bw()

####
plot6864 <- ggplot(veg.v6800.6401.long, aes(fill=specie, y=value, x=Area)) + 
  geom_bar(position="fill", stat="identity") +
  scale_fill_manual(values = c("azure3", "darkseagreen","goldenrod","goldenrod1","lemonchiffon",
                               "darkolivegreen3","goldenrod4","darkseagreen2","darkseagreen3")) + 
  ggtitle("6800-6400 cal. BP") +
  theme(axis.title.x = element_blank(),
        axis.text.x  = element_text(angle = 90)) +
  theme_bw()


total.plot.merge3 <- ggpubr::ggarrange(plot7674, plot7472, plot7268, plot6864,
                                       ncol = 1, nrow = 4, common.legend = TRUE)
total.plot.merge3

ggarrange(total.plot.merge3, renyi.diacronico,
          ncol = 2, nrow = 1)











